//*************************************************************************************************
//
//	Description:
//		vehicles_basic.fx - Version of basic.fx which should be used on vehicles, has the option
//															of a dirt/damage/wetness channel, plus more versatile envmapping.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     01/25/2008  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_SPECULAR
#define NORMAL_MAPPING
#define ENV_MAPPING
#define USE_FRESNEL
#define ALCANTARA
#define DIRT_SCRATCH
#define USE_LIGHT_CONTROL
#endif

#define MAX_ENVMAP_BIAS	6.0f

// Uncomment to use experimental higher quality envmapping approach
#define EXPERIMENTAL_ENVMAPPING



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#elif defined( ENV_MAPPING )
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

// Used for the dirt and noise mapping - must be no overlaps
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = 4;
	int RuntimeTexcoord = 3;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Used for the emissive lighting control
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 5;
	int RuntimeTexcoord = 4;
	bool ProtectFromRescaling = true;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

SPECIALISATION_PARAM( useFresnel, "Use Fresnel?", "USE_FRESNEL" )	// TRUE if fresnel factor should be used

#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fresnelFactor, "Fresnel Factor", useFresnel, 0.0f, 1.0f, 0.6f )
#endif

SPECIALISATION_PARAM( alcantara, "Alcantara?", "ALCANTARA" )	// TRUE if alcantara effect should be applied

#if defined( _3DSMAX_ ) || defined( ALCANTARA )
DEPENDENT_COLOUR_PARAM( alcantaraColour, "Alcantara Rim Col", alcantara )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 65536.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 65536.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 16.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 16.0f, 1.0f )
#endif


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

//
// Dirt n scratches
//

SPECIALISATION_PARAM( dirtNscratch, "Dirt 'n' Scratch?", "DIRT_SCRATCH" )	// TRUE if dirt and scratches should be applied
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord4Dependency, texCoord3, TEXCOORD3, dirtNscratch )

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
DEPENDENT_TEXTURE_PARAM( scratchControlTexture, "Dirt / Scratch Tex.", dirtNscratch )			// multi-channel dirt and scratch texture, will be supplied by game
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( primerBasis, "Primer Colour", dirtNscratch, 0.7f, 0.7f, 0.7f, 1.0f  )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( metalBasis,  "Metal Colour",  dirtNscratch, 0.3f, 0.3f, 0.3f, 1.0f  )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( dirtBasis,  "Dirt Colour",  dirtNscratch, 0.233f, 0.203f, 0.100f, 1.0f  )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( noiseScale, "Dirt Noise Scale", dirtNscratch, 0.0f, 128.0f, 4.0f )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( dirtScale,  "Set me blue",  dirtNscratch, 0.0f, 0.0f, 1.0f, 1.0f  )
#endif


// TRUE if the material represents vehicle lights & so self-emits under code control
SPECIALISATION_PARAM( useLightControl, "Light Control?", "USE_LIGHT_CONTROL" )
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord5Dependency, texCoord4, TEXCOORD4, useLightControl )

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
DEPENDENT_TEXTURE_PARAM( emissiveTexture, "Emissive {UV1}", useLightControl )

SPECIALISATION_PARAM( allowBroken, "Allow Broken?", "BROKEN_LIGHT" )
DEPENDENT_TEXTURE_PARAM( replacementAlphaTex, "Broken Alpha Tex", allowBroken )

#define NUM_EMISSION_CONTROL_VALUES		20
float emissionControl[ NUM_EMISSION_CONTROL_VALUES ]
<
	string UIName = "emissionControl";
	bool appEdit = true;
	bool export = false;
	string exportDependsOn = "useLightControl";
>;

float testLightLevel
<
	string UIName = "Test Light Level";
	bool appEdit = false;
	bool export = false;
	float UIMin = -1.0f;		
	float UIMax = 10.0f;                                  
> = 0.0f;
#endif



// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined ( DIRT_SCRATCH )
sampler2D scratchControlMap : sample
<
	SET_LINEAR_TEXTURE
	bool appEdit = false;
	string SamplerTexture="scratchControlTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < scratchControlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined ( USE_LIGHT_CONTROL )
sampler2D emissiveMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="emissiveTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < emissiveTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
sampler2D brokenAlphaMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="replacementAlphaTex"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < replacementAlphaTex >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif // BROKEN_LIGHT

#endif


//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float CalculateFresnel( float3 _eye, float3 _normal, float rindexRatio )
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}



// Function calculates emissiveness by looking up a value from an array which the app sets,
// indexed by the control ID passed. If a negative ID is passed, the emission is considered to
// be always on.
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
float CalcEmissiveness( float _ID, float _brightness )
{
	int intControlID = floor( _ID );

	if ( intControlID>=0 && intControlID<NUM_EMISSION_CONTROL_VALUES )
	{
#if defined( _3DSMAX_ )
		return testLightLevel * abs( _brightness );
#else
		return min( emissionControl[ intControlID ] * abs( _brightness ), 2.0f );
#endif
	}
	else
	{
		return 0.0f;
	}
}
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
	float2 texCoord3 : TEXCOORD3;													// UV channel 4 texture coord (dirt n scratch map)
	float2 texCoord4 : TEXCOORD4;													// UV channel 5 texture coord (emissive control)
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#if defined( DIRT_SCRATCH )
  float2 texCoord3 : TEXCOORD3;
#endif
#if defined( USE_LIGHT_CONTROL )
	float2 texCoord4 : TEXCOORD4;													// UV channel 5 texture coord (emissive control)
#endif
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// Texture coords
};

// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	float4 texCoord23	: TEXCOORD0;												// UV coords for texture channel 0 + dirt channel coords
#else
	float2 texCoord23	: TEXCOORD0;												// UV coords for texture channel 0
#endif
	float4 normal_and_fresnel	: TEXCOORD1;								// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float2 emissiveness : TEXCOORD3;												// Emissiveness amount in x, and broken state in Y (0.0 if not used)
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
#endif
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 coords			: TEXCOORD1;
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT VehicleBasicVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
#if defined( _3DSMAX_ )
	_output.texCoord23 = float4(_input.texCoord.x, _input.texCoord.y, _input.texCoord3.x, 1.0f - _input.texCoord3.y ) ;
#else
	_output.texCoord23.xy = _input.texCoord;
	#if defined ( DIRT_SCRATCH )
		_output.texCoord23.zw = _input.texCoord3.xy;
	#endif
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )

		_output.emissiveness.x = CalcEmissiveness( _input.texCoord4.x, _input.texCoord4.y );

#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
	_output.emissiveness.y = saturate( sign( -_output.emissiveness.x) );
#else
	_output.emissiveness.y = 0.0f;
#endif
		_output.emissiveness.x = max( _output.emissiveness.x, 0.0f );
#endif
	DEPENDENT_CODE_END( useLightControl )

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// If its required
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		// Calculate fresnel term
		_output.normal_and_fresnel = float4( normal, CalculateFresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		// Fresnel term is 1.0
		_output.normal_and_fresnel = float4( normal, 1.0f );
#endif
	DEPENDENT_CODE_END( useFresnel )

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



SHADOWGEN_VSOUTPUT VehicleBasicTranslucentShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

#if defined(NEEDS_SHADOW_COORDS)
	OUTPUT_SHADOW_COORDS;
#endif

	return _output;
}


ZPRIMEDOF_VSOUTPUT VehicleBasicTranslucentZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			  : TEXCOORD4;							// Vertex colour
	float4 texCoord23	  : TEXCOORD0;						// UV coords for texture channel 0 and dirt
	float4 normal_and_fresnel	: TEXCOORD1;		// Normal vector (world space) + fresnel coefficient
	float4 eye				  : TEXCOORD2;						// Eye vector (world space)
	float2 emissiveness : TEXCOORD3;						// Emissiveness amount and brokenness amount
	float3 tangent		  : TEXCOORD5;						// Tangent vector (world space)
	float3 binormal		  : TEXCOORD6;						// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD4;											// Vertex colour
#if defined( DIRT_SCRATCH )	
	float4 texCoord23	: TEXCOORD0;									// UV coords for texture channel 0 and dirt
#else
	float2 texCoord23	: TEXCOORD0;									// UV coords for base texture
#endif
	float4 normal_and_fresnel	: TEXCOORD1_centroid;	// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2_centroid;					// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float2 emissiveness : TEXCOORD3_centroid;				// Emissiveness amount and brokenness amount
#endif

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;					// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;					// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


struct SHADOWGEN_PSINPUT
{
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
#if defined(NEEDS_SHADOW_COORDS)
	float4 shadowCoord	: TEXCOORD1;
#endif
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT VehicleBasicFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord23.xy );
	float4 specularTexColour;
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord23.xy );

	#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
	float replacementAlpha = 1.0f; 
	float finalScale = 1.0f;
	DEPENDENT_CODE_START( allowBroken )	
		replacementAlpha = tex2D( brokenAlphaMap, _input.texCoord23.xy ).y;	// using green channel as it is marginally more accurate
		finalScale = lerp( diffuseTexColour.a, replacementAlpha, _input.emissiveness.y );
	DEPENDENT_CODE_END( allowBroken )
#endif
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	float	globalEMapFactorValue = globalEMapFactor;
#endif

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	float4 controlColour;
	float paintPercent;
	float glossPercent;
	float metalPercent;
	float primerPercent;
	float specularPercent;
	float dirtPercent;
	float fresnelPercent;
  float2 scratchControlUV;	// used to simplify code, to reduce risk of errors
#endif

	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )

	scratchControlUV.xy = _input.texCoord23.zw;

	controlColour = tex2D( scratchControlMap, scratchControlUV.xy );	// get control map colours 
	
	paintPercent = saturate( (0.34f - controlColour.g) * 50.0f );	// note these values are interdependent - don't change them without speaking to RobD
	glossPercent = saturate( 1.00f - (controlColour.g * 10.0f) );	// gloss value drops sooner, so that we appear to scrape through the surface before going through the paint
	metalPercent = saturate( (controlColour.g - 0.65f) * 50.0f ); 
	primerPercent = (1.0f - (paintPercent + metalPercent));

	specularPercent = glossPercent + metalPercent * metalBasis.w;	// using metal colour alpha channel to control shininess of metal layer
	
	dirtPercent = controlColour.b;	// no messing with this one, its just as it is, a splotch of dirt, with variable thickness

	specularPercent *= smoothstep( 1.0f, 0.0f, dirtPercent * 4.0f );	// scale down specular if there is any dirt on it

	fresnelPercent = glossPercent;	// probably optimises out, but done this way for clarity
#endif
	DEPENDENT_CODE_END( dirtNscratch )

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord23.xy );
#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to ones (will be optimised out)
		specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	// apply dirt and scratches
	float4 interimColour = lerp( diffuseTexColour, primerBasis, primerPercent);
	interimColour = lerp( interimColour, metalBasis, metalPercent);
	diffuseTexColour = lerp( interimColour, dirtBasis, dirtPercent);

	globalSpecularFactorValue *= specularPercent;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	DEPENDENT_CODE_START( useEnvironmentMap )
		globalEMapFactorValue *= specularPercent;
	DEPENDENT_CODE_END( useEnvironmentMap )
#endif

#endif
  DEPENDENT_CODE_END( dirtNscratch )
  	
  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal_and_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord23.xy );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	//
	// Calculate the fresnel factor
	//
	float fresnelCoefficient;
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		fresnelCoefficient = saturate( _input.normal_and_fresnel.w );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		fresnelCoefficient = 1.0f;
#endif
	DEPENDENT_CODE_END( useFresnel )

	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
//	fresnelCoefficient *= fresnelPercent;
#endif
  DEPENDENT_CODE_END( dirtNscratch )

	// If the alcantara effect is required
	DEPENDENT_CODE_START( alcantara )
#if defined( _3DSMAX_ ) || defined( ALCANTARA )
		// Lerp the diffuse colour towards mid-grey based on the fresnel factor
		diffuseTexColour = lerp( diffuseTexColour, float4( alcantaraColour.r, alcantaraColour.g, alcantaraColour.b, diffuseTexColour.a ), fresnelCoefficient );
#endif
  DEPENDENT_CODE_END( alcantara )

#if defined( PSEUDO_GI )

	// If environment mapping is switched on
	float4 accumulatedColour;
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate ambient-lit base colour (looking up the envmap for the ambient lighting)
		float4 envmapambient = texCUBElod( environmentMap, float4( normal.xyz, 4 ) );
		accumulatedColour = diffuseTexColour * envmapambient;
#endif
	DEPENDENT_CODE_ELSE( useEnvironmentMap )
#if defined( _3DSMAX_ ) || !defined( ENV_MAPPING )
		// Calculate ambient-lit base colour from interpolated lighting
		accumulatedColour = diffuseTexColour * _input.colour;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

#else
	float4 accumulatedColour = diffuseTexColour * _input.colour;
#endif

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// Add in emissive lighting
	accumulatedColour.rgb += _input.emissiveness.x * emissiveColour.rgb * emissiveColour.a;
#endif
	DEPENDENT_CODE_END( useLightControl )

	//
	// If we're dirt mapping, perturb the normal back towards the tangent space one
	// according to the dirt amount.
	// 
	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
		normal = normalize( lerp( normal, TSnormal, saturate( dirtPercent * 4.0f ) ) );
#endif
  DEPENDENT_CODE_END( dirtNscratch )

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

#if defined( EXPERIMENTAL_ENVMAPPING )
	// Intersect the reflection vector with a 4m sphere centred on the car
	float3 envMapSpacePos = mul( float4( worldPos, 1.0f ), envTransformI ).xyz;
	float3 rayOrigin = envMapSpacePos;
	float3 rayDirection = reflectionVector;
	float B = dot( rayDirection, rayOrigin ) * 2.0f;
	float C = dot( rayOrigin, rayOrigin ) - ( 4.0f * 4.0f );
	float t0 = -B - sqrt( B*B - 4.0f * C ) / 2.0f;
	float t1 = -B + sqrt( B*B - 4.0f * C ) / 2.0f;
	float3 sphereIntersection = rayOrigin + ( rayDirection * max( t0, t1 ) );

	// Use the intersection point to look up the cubemap, not the reflection vector
	reflectionVector = normalize( sphereIntersection );
#endif

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactorValue * fresnelCoefficient;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;

#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
	accumulatedColour *= finalScale;
#endif

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float2 emissiveness : TEXCOORD2;										// Emissiveness amount
#endif
};


VSOUTPUT_LD VehicleBasicLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
		_output.emissiveness.x = CalcEmissiveness( _input.texCoord4.x, _input.texCoord4.y );
		
#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
		_output.emissiveness.y = saturate( sign( -_output.emissiveness.x) );
#else
		_output.emissiveness.y = 0.0f;
#endif

		_output.emissiveness.x = max( _output.emissiveness.x, 0.0f );

		// Calculate world-space vector to the eye
#ifdef _3DSMAX_
		float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
		float3 worldEyeVec = worldCameraPos - worldPos;
#endif
#endif
	DEPENDENT_CODE_END( useLightControl )

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT VehicleBasicLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
	float replacementAlpha = 1.0f; 
	float finalScale = 1.0f;
	DEPENDENT_CODE_START( allowBroken )	
		replacementAlpha = tex2D( brokenAlphaMap, _input.texCoord.xy ).y;	// using green channel as it is marginally more accurate
		finalScale = lerp( diffuseTexColour.a, replacementAlpha, _input.emissiveness.y );
	DEPENDENT_CODE_END( allowBroken )
#endif

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// Add in emissive lighting
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord );
	accumulatedColour.rgb += _input.emissiveness.x * emissiveColour.rgb * emissiveColour.a;
#endif
	DEPENDENT_CODE_END( useLightControl )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

#if defined( _3DSMAX_ ) || defined( BROKEN_LIGHT )
	_output.Colour *= finalScale;
#endif

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT VehicleBasicTranslucentShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

#if defined(NEEDS_SHADOW_COORDS)
	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
#else
	output.Colour = 0;
#endif
	output.Colour.a = diffuseTexColour.a * _input.colour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.25f );

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT VehicleBasicTranslucentZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = diffuseTexColour.a * _input.colour.a;

	return output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique VehicleBasic
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "VehicleBasic";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "VehicleBasic_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx VehicleBasicVertexShader();
		PixelShader = compile sce_fp_rsx VehicleBasicFragmentShader();
#else
		VertexShader = compile vs_3_0 VehicleBasicVertexShader();
		PixelShader = compile ps_3_0 VehicleBasicFragmentShader();
#endif
	}
}



technique VehicleBasic_Translucent
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "VehicleBasic";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_VehicleBasic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "VehicleBasic_LowDetail";
	int    lowDetailDeferredID	= 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 VehicleBasicVertexShader();
		PixelShader = compile ps_3_0 VehicleBasicFragmentShader();
#endif
	}
}


technique VehicleBasic_AlphaTest
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "VehicleBasic";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_VehicleBasic_Translucent_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_VehicleBasic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "VehicleBasic_LowDetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = true;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif


#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 VehicleBasicVertexShader();
		PixelShader = compile ps_3_0 VehicleBasicFragmentShader();
#endif
	}
}



technique VehicleBasic_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "VehicleBasic_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx VehicleBasicLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx VehicleBasicLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 VehicleBasicLowDetailVertexShader();
		PixelShader = compile ps_3_0 VehicleBasicLowDetailFragmentShader();
#endif
	}
}


technique VehicleBasic_Translucent_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "VehicleBasic_LowDetail";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_VehicleBasic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 VehicleBasicVertexShader();
		PixelShader = compile ps_3_0 VehicleBasicFragmentShader();
#endif
	}
}


//
// This technique only differs from VehicleBasic by the renderstates
// it requires, so its a max-only technique.
//
technique VehicleBasic_AlphaTest_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "VehicleBasic_LowDetail";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_VehicleBasic_Translucent_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_VehicleBasic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 VehicleBasicVertexShader();
		PixelShader = compile ps_3_0 VehicleBasicFragmentShader();
#endif
	}
}



technique _VehicleBasic_Translucent_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 0x40 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx VehicleBasicTranslucentShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx VehicleBasicTranslucentShadowGenFragmentShader();
#elif defined (_XBOX)
			VertexShader = compile vs_3_0 VehicleBasicTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 VehicleBasicTranslucentShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 VehicleBasicTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 VehicleBasicTranslucentShadowGenFragmentShader();
#endif
	}
}

technique _VehicleBasic_Translucent_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 0x40);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx VehicleBasicTranslucentZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx VehicleBasicTranslucentZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 0x40;
		  AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 VehicleBasicTranslucentZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 VehicleBasicTranslucentZPrimeDOFFragmentShader();
#endif
	}
}
